/****************************************************************************
 *
 * CRI Middleware SDK
 *
 * Copyright (c) 2013-2018 CRI Middleware Co., Ltd.
 *
 * Library  : CRIWARE plugin for Unreal Engine 4
 * Module   : AudioComponent for Atom
 * File     : AtomComponent.cpp
 *
 ****************************************************************************/

/***************************************************************************
 *      インクルードファイル
 *      Include files
 ***************************************************************************/
/* モジュールヘッダ */
#include "AtomComponent.h"

/* CRIWAREプラグインヘッダ */
#include "CriWareRuntimePrivatePCH.h"
#include "AtomSoundObject.h"
#include "AtomActiveSound.h"
#include "AtomPlaylist.h"
#include "AtomListener.h"
#include "AtomParameterComponent.h"

/* Unreal Engine 4関連ヘッダ */
#include "Engine/Engine.h"
#include "Application/ThrottleManager.h"
#include "AudioThread.h"
#include "UObject/Package.h"
#include "Misc/PackageName.h"
#include "Engine/LocalPlayer.h"
#include "UnrealClient.h"

#if WITH_EDITOR
#include "DrawDebugHelpers.h"
#endif

/***************************************************************************
 *      定数マクロ
 *      Macro Constants
 ***************************************************************************/
#define LOCTEXT_NAMESPACE "AtomComponent"

/***************************************************************************
 *      処理マクロ
 *      Macro Functions
 ***************************************************************************/

/***************************************************************************
 *      データ型宣言
 *      Data Type Declarations
 ***************************************************************************/

/***************************************************************************
 *      変数宣言
 *      Prototype Variables
 ***************************************************************************/

/***************************************************************************
 *      クラス宣言
 *      Prototype Classes
 ***************************************************************************/

/***************************************************************************
 *      関数宣言
 *      Prototype Functions
 ***************************************************************************/

/***************************************************************************
 *      変数定義
 *      Variable Definition
 ***************************************************************************/
int32 UAtomComponent::AtomComponentIDCounter = 0;
TMap<int32, UAtomComponent*> UAtomComponent::AtomIDToComponentMap;
TArray<UAtomComponent*> UAtomComponent::AtomNonRegisteredComponentArray;
TArray<UAtomComponent*> UAtomComponent::AtomRootedComponentArray;

namespace
{
	static bool bEnableDefaultAttenuation = true;
}

/***************************************************************************
 *      クラス定義
 *      Class Definition
 ***************************************************************************/
FAtomAisacControlParam::FAtomAisacControlParam()
	: Value(0.0f)
{
#if WITH_EDITOR
	/* Localization of unreal properties metadata with LOCTEXT markups and reflection */
	CRI_LOCSTRUCT(StaticStruct());
#endif
}

FAtomSelectorParam::FAtomSelectorParam()
{
#if WITH_EDITOR
	/* Localization of unreal properties metadata with LOCTEXT markups and reflection */
	CRI_LOCSTRUCT(StaticStruct());
#endif
}

UAtomComponent::UAtomComponent(const FObjectInitializer& ObjectInitializer)
	: Super(ObjectInitializer)
	, AtomComponentID(0)
	, Filter_Frequency(MAX_FILTER_FREQUENCY)
{
#if WITH_EDITOR
	/* Localization of unreal properties metadata with LOCTEXT markups and reflection */
	CRI_LOCCLASS(GetClass());
#endif

	/* AtomComponent初期化時にリセットするパラメータ */
	PrimaryComponentTick.bCanEverTick = true;
	PrimaryComponentTick.bStartWithTickEnabled = true;
	bTickInEditor = true;

	bAutoDestroy = false;
	bPersistAcrossLevelTransition = false;
	bAddedAtomNonRegisteredComponentArray = false;
	bAutoActivate = true;
	bAllowAnyoneToDestroyMe = true;
	bStopWhenOwnerDestroyed = true;
	bIsUISound = false;
	bIsPreviewSound = false;
	DefaultVolume = 1.0f;
	bEnableMultipleSoundPlayback = false;
	bUsePlaylist = false;
	DefaultSoundObject = nullptr;
	DefaultBlockIndex = 0;
	DefaultAisacControl.Empty();
	DefaultSelectorLabel.Empty();
	Player = NULL;
	Source = NULL;
	Listener = nullptr;
	DistanceFactor = 1.0f;
	CueSheet = NULL;
	AppliedSoundObject = nullptr;
	PlaybackId = 0xFFFFFFFF;
	bIsPlayed = false;
	Playlist = nullptr;
	AtomComponentID = 0;

	/* 距離減衰の上書きをデフォルト状態では無効化 */
	AttenuationSettings = nullptr;
	bOverrideAttenuation = false;
	AttenuationOverrides.bAttenuate = false;
	AttenuationOverrides.bSpatialize = false;

	/* 音源位置更新のため、OnUpdateTransformを呼び出してもらうよう指定する。 */
	bWantsOnUpdateTransform = true;

	/* 不要な機能を無効化 */
	bNeverNeedsRenderUpdate = true;
	bNavigationRelevant = false;
	bVisible = false;

#if WITH_EDITORONLY_DATA
	/* 音源のスピーカアイコン表示を有効にする */
	bVisualizeComponent = true;
#endif

	/* デリゲート登録のリセット */
	OnWorldCleanedUpHandle.Reset();

	CreatedTime = FPlatformTime::Seconds();

	PreviousSourceLocation.x = 0.0f;
	PreviousSourceLocation.y = 0.0f;
	PreviousSourceLocation.z = 0.0f;
	PreviousListenerLocation.x = 0.0f;
	PreviousListenerLocation.y = 0.0f;
	PreviousListenerLocation.z = 0.0f;
	bTransformUpdated = false;
	bIsEnableDoppler = false;

	/* Componentのタグ付け */
	this->ComponentTags.Add(TEXT("AtomComponent"));

	/* 作成時に一意のIDを付与 */
	if ((AtomComponentIDCounter == MAX_int32) || (AtomComponentIDCounter < 0)) {
		AtomComponentIDCounter = 0;
	}
	AtomComponentID = ++AtomComponentIDCounter;

	/* ComponentをMapに登録 */
	if (!HasAnyFlags(RF_ClassDefaultObject)) {
		AtomIDToComponentMap.Add(AtomComponentID, this);
	}
}

void UAtomComponent::MarkAsRooted()
{
	AddToRoot();
	AtomRootedComponentArray.Add(this);
	bPersistAcrossLevelTransition = true;
}

/* 非レジスターAtomComponentのTick */
void UAtomComponent::TickNonRegisteredComponents()
{
	TArray<UAtomComponent*> ArrayForIteration = AtomNonRegisteredComponentArray;
	for (UAtomComponent* AtomComponent : ArrayForIteration) {
		if (!IsValid(AtomComponent)) {
			continue;
		}
		if (AtomComponent->IsBeingDestroyed()) {
			continue;
		}
		/* Ownerの座標を使ってUpdateTransformする */
		AtomComponent->UpdateTransform(true);
		AtomComponent->OnTickComponent();
	}
}

/* ルートオブジェクトのTick */
void UAtomComponent::TickRootedComponents()
{
	TArray<UAtomComponent*> ArrayForIteration = AtomRootedComponentArray;
	for (UAtomComponent* AtomComponent : ArrayForIteration) {
		if (!IsValid(AtomComponent)) {
			continue;
		}
		if (AtomComponent->IsBeingDestroyed()) {
			continue;
		}
		AtomComponent->OnTickComponent();
	}
}

void UAtomComponent::SetSound(USoundAtomCue* NewSound)
{
	/* キューシートの取得 */
	USoundAtomCueSheet* NewCueSheet = ((NewSound != nullptr) ? NewSound->CueSheet : nullptr);

	/* キュー名の取得 */
	FString NewCueName = ((NewSound != nullptr) ? NewSound->CueName : "");

	/* データの指定 */
	SetSound(NewCueSheet, NewCueName);

	/* サウンドデータの更新 */
	Sound = NewSound;
}

void UAtomComponent::SetSound(USoundAtomCueSheet* InCueSheet, FString InCueName)
{
	/* 再生中の音声を停止 */
	if (bEnableMultipleSoundPlayback == false) {
		Stop();
	}

	/* 既存サウンドデータのクリア */
	Sound = nullptr;

	/* キューシートの変更チェック */
	if (CueSheet == InCueSheet) {
		/* データが変更されていない場合はリロード不要 */
	} else {
		/* 既存キューシートを一旦破棄 */
		if (CueSheet != nullptr) {
			CueSheet->Release();
			CueSheet = nullptr;
		}

		/* キューシートのリロード */
		CueSheet = InCueSheet;
		if (CueSheet != nullptr) {
			CueSheet->AddRef();
		}
	}

	/* キュー名の保存 */
	CueName = InCueName;

	/* プレイリストのクリア */
	if (Playlist != nullptr) {
		Playlist->Reset();
	}
}

/* 音声データのセット */
void UAtomComponent::EnqueueSound(USoundAtomCue* NewSound)
{
	/* プレイリスト再生が有効かどうかをチェック */
	if (bUsePlaylist == false) {
		return;
	}

	/* シームレス連結再生用にプレイリストを作成 */
	/* 注意）再生開始前にEnqueueSoundは実行される可能性あり。 */
	if (Playlist == nullptr) {
		Playlist = new FAtomPlaylist();
	}

	/* プレイリストに追加 */
	if ((Playlist != nullptr) && (NewSound != nullptr)) {
		Playlist->Add(NewSound);
	}
}

int32 UAtomComponent::GetNumQueuedSounds()
{
	if (Playlist == nullptr) {
		return 0;
	}

	/* 排他制御区間の開始 */
	Playlist->Lock();

	/* キューに残っているサウンドの数を取得 */
	int32 NumQueued = Playlist->Num();

	/* 再生位置の取得 */
	int32 Pos = Playlist->Tell();

	/* 排他制御区間の終了 */
	Playlist->Unlock();

	/* 再生済みの音声を除外 */
	return (NumQueued - Pos);
}

void UAtomComponent::Play(float StartTime)
{
	PlayInternal(StartTime, 0.0f, 1.0f);
}

void UAtomComponent::PlayInternal(float StartTime, float FadeInDuration, float FadeVolumeLevel)
{
	/* 既に再生中の場合は一旦停止 */
	if (bEnableMultipleSoundPlayback == false) {
		Stop();
	}

	/* 負値のシークは無視 */
	int32 StartTimeMs = (int32)(StartTime * 1000.0f);
	if (StartTimeMs < 0) {
		/* 備考）シーケンサエディタ上で再生カーソルを			*/
		/* 　　　波形の先頭以前に移動した場合にここに到達する。	*/
		return;
	}

	/* 再生用のリソースが確保済みかどうかチェック */
	if (Player == NULL) {
		/* リソースの確保 */
		AllocateResource();

		/* 発音が可能かどうかチェック */
		if (Player == NULL) {
			/* プレーヤが存在しない場合は再生不可 */
			return;
		}
	}

#if WITH_EDITOR
	/* ゲーム実行中かどうかチェック */
	UWorld* World = GetWorld();
	bool bIsGameWorld = (World ? World->IsGameWorld() : false);

	/* 発音の可否を判定 */
	/* 備考）エディタ上では以下のケースのみ発音を行う。             */
	/* 　　　- プレビュー実行中。                                   */
	/* 　　　- （PersonaやMatineeで）UIサウンドとして発音する場合。 */
	if ((GIsEditor != false) && (bIsGameWorld == false) && (bIsUISound == false)) {
		return;
	}

	/* 既存デリゲート登録の解除 */
	if (OnWorldCleanedUpHandle.IsValid() != false) {
		FWorldDelegates::OnWorldCleanup.Remove(OnWorldCleanedUpHandle);
		OnWorldCleanedUpHandle.Reset();
	}

	/* プレビュー終了検知用にデリゲートを登録 */
	OnWorldCleanedUpHandle = FWorldDelegates::OnWorldCleanup.AddUObject(this, &UAtomComponent::OnWorldCleanedUp);
#endif

	/* キューシート、キュー名の補完 */
	if (Sound != nullptr) {
		/* キューシートのロード */
		/* 注意）マップ上に配置された音はSetSoundを経由しないため、 */
		/* 　　　この時点でCueSheetがnullptrの可能性あり。          */
		if (CueSheet == nullptr) {
			CueSheet = Sound->CueSheet;
			if (CueSheet != nullptr) {
				CueSheet->AddRef();
			}
		}

		/* 互換性維持のための処理 */
		/* 備考）2014/03以前のプラグインで作成したデータは、	*/
		/* 　　　CueSheetの参照ではなくCueSheetNameを使用。		*/
		if ((CueSheet == nullptr) && (Sound->CueSheetName.Len() > 0)) {
			FString PackagePath = FPackageName::GetLongPackagePath(Sound->GetOutermost()->GetName());
			CueSheet = USoundAtomCueSheet::Find(*Sound->CueSheetName, PackagePath);
			if (CueSheet != nullptr) {
				CueSheet->AddRef();
			}
		}

		/* キュー名の補完 */
		if (CueName.Len() <= 0) {
			CueName = Sound->CueName;
		}
	}

	FString SelectedCueName = CueName;
	USoundAtomCueSheet* SelectedCueSheet = CueSheet;

	/* キュー名がプレイリストから補完可能かチェック */
	if ((Playlist != nullptr) && (SelectedCueName.Len() <= 0)) {
		/* 排他制御区間の開始 */
		Playlist->Lock();

		/* 再生位置の取得 */
		int32 Pos = Playlist->Tell();

		/* リストアイテムの取得 */
		const FAtomPlaylistItem* Item = Playlist->Get(Pos);
		if ((Item != nullptr) && (Item->Sound != nullptr)) {
			/* キューの情報を取得 */
			SelectedCueName = Item->Sound->CueName;
			SelectedCueSheet = Item->Sound->CueSheet;

			/* 再生位置の更新 */
			Playlist->Seek(Pos + 1);
		}

		/* 排他制御区間の終了 */
		Playlist->Unlock();
	}

	/* キュー名が指定されていない場合は何もしない */
	if (SelectedCueName.Len() <= 0) {
		return;
	}

	/* キュー名の設定 */
	CriAtomExAcbHn AcbHn = ((SelectedCueSheet != nullptr) ? SelectedCueSheet->GetAcbHn() : nullptr);
	criAtomExPlayer_SetCueName(Player, AcbHn, TCHAR_TO_UTF8(*SelectedCueName));

	/* 再生パラメータの設定*/
	criAtomExPlayer_SetVolume(Player, DefaultVolume);
	criAtomExPlayer_SetFirstBlockIndex(Player, DefaultBlockIndex);
	criAtomExPlayer_SetStartTime(Player, StartTimeMs);
	if (FadeInDuration > 0.0f) {
		float AttackTime = FMath::Min(FadeInDuration * 1000.0f, 2000.0f);
		criAtomExPlayer_SetEnvelopeAttackTime(Player, AttackTime);
	}
	for (int32 i = 0; i < DefaultAisacControl.Num(); i++) {
		const FAtomAisacControlParam& P = DefaultAisacControl[i];
		criAtomExPlayer_SetAisacControlByName(Player, TCHAR_TO_UTF8(*P.Name), P.Value);
	}
	for (int32 i = 0; i < DefaultSelectorLabel.Num(); i++) {
		const FAtomSelectorParam& P = DefaultSelectorLabel[i];
		criAtomExPlayer_SetSelectorLabel(Player, TCHAR_TO_UTF8(*P.Selector), TCHAR_TO_UTF8(*P.Label));
	}

	/* 音源の位置を更新 */
	UpdateTransform();

	/* 距離減衰と3Dポジショニングの上書き判定 */
	if (((bOverrideAttenuation != false) || (AttenuationSettings != NULL) || (Sound && Sound->DefaultAttenuation)) && (bEnableDefaultAttenuation == true)) {
		const auto* Settings = GetAttenuationSettingsToApply();

		/* Non-Spatialized radius の設定を反映する */
		if ((Settings != NULL) && (Source != NULL)) {
			criAtomEx3dSource_SetInteriorPanField(Source, Settings->OmniRadius, Settings->OmniRadius);
		}

		/* 距離減衰上書き時はADX2側の距離減衰計算を無効にする */
		criAtomEx3dSource_SetAttenuationDistanceSetting(Source, CRI_FALSE);
		criAtomEx3dSource_Update(Source);

		if ((Settings != NULL) && (Settings->bSpatialize == true)) {
			criAtomExPlayer_SetPanType(Player, CRIATOMEX_PAN_TYPE_3D_POS);
		} else {
			criAtomExPlayer_SetPanType(Player, CRIATOMEX_PAN_TYPE_PAN3D);
		}

		/* 減衰ボリュームの計算 */
		float Volume = DefaultVolume;
		ApplyAttenuation(Volume, Filter_Frequency);

		/* ボリュームの更新 */
		criAtomExPlayer_SetVolume(Player, Volume);

		/* LPFの適用 */
		criAtomExPlayer_SetBandpassFilterParameters(Player, 0.f, Filter_Frequency / MAX_FILTER_FREQUENCY);
	}

	/* AtomParameterの適用 */
	UAtomParameterComponent* AtomParameter = nullptr;
	if ((DefaultSoundObject == nullptr) && (Sound != nullptr) && (Sound->bApplyAtomParameter != false)) {
		/* パラメータの検索 */
		AActor *TestActor = GetOwner();
		while (TestActor) {
			/* Actorにパラメータが設定されているかどうかチェック */
			AtomParameter = Cast<UAtomParameterComponent>(
				TestActor->GetComponentByClass(UAtomParameterComponent::StaticClass()));
			if (AtomParameter != nullptr) {
				break;
			}

			TestActor = TestActor->GetOwner();
		}
	}

	/* サウンドオブジェクトの適用 */
	UAtomSoundObject* SoundObject = (AtomParameter != nullptr) ? AtomParameter->SoundObject : DefaultSoundObject;
	ApplySoundObject(SoundObject);

#if WITH_EDITOR
	/* エディター上（非プレビュー中）のみ鳴らす音はパン3Dで再生 */
	/* 注意）シーケンサーで再生する音は常にUIサウンドとして再生されるため、	*/
	/* 　　　UIサウンドであっても暗黙でパン3Dを適用してはいけない。			*/
	if ((bIsGameWorld == false) && (bIsUISound == true)) {
		criAtomExPlayer_SetPanType(Player, CRIATOMEX_PAN_TYPE_PAN3D);
	}
#endif

	/* DopplerFactorが設定されているか確認 */
	bIsEnableDoppler = false;
	if (AcbHn != nullptr) {
		CriAtomExCueInfo AtomCueInfo;
		CriBool isSuccess = criAtomExAcb_GetCueInfoByName(static_cast<CriAtomExAcbHn>(AcbHn), TCHAR_TO_UTF8(*SelectedCueName), &AtomCueInfo);
		if (isSuccess == CRI_TRUE && AtomCueInfo.pos3d_info.doppler_factor != 0.0f) {
			bIsEnableDoppler = true;
		}
	}

	/* 再生の開始 */
	PlaybackId = criAtomExPlayer_Start(Player);

	/* 再生時のLocationを保持しておく */
	FVector tmp_location = GetComponentLocation();
	PreviousSourceLocation.x = tmp_location.X * DistanceFactor;
	PreviousSourceLocation.y = tmp_location.Y * DistanceFactor;
	PreviousSourceLocation.z = tmp_location.Z * DistanceFactor;

	/* 再生の開始を通知 */
	bIsPlayed = true;

	/* アクティブ化の通知 */
	bIsActive = true;

#if WITH_EDITOR
	/* Play In Editorモードで減衰距離を表示するため、キュー情報から取得 */
	if (GIsPlayInEditorWorld && MaxDistanceSphereComponent && MinDistanceSphereComponent) {
		if (AcbHn != nullptr) {
			CriAtomExCueInfo cueInfo = { 0 };
			CriBool isGetCueInfoSucceed = CRI_FALSE;
			if (criAtomExAcb_ExistsName(static_cast<CriAtomExAcbHn>(AcbHn), TCHAR_TO_UTF8(*(CueName))) == CRI_TRUE) {
				isGetCueInfoSucceed = criAtomExAcb_GetCueInfoByName(static_cast<CriAtomExAcbHn>(AcbHn), TCHAR_TO_UTF8(*(CueName)), &cueInfo);
			}

			MaxDistanceSphereComponent->SetSphereRadius(cueInfo.pos3d_info.max_distance / DistanceFactor, false);
			MinDistanceSphereComponent->SetSphereRadius(cueInfo.pos3d_info.min_distance / DistanceFactor, false);
		}
	}
#endif
}

/* データ要求コールバック */
void CRIAPI UAtomComponent::OnDataRequest(
	void *Object, CriAtomExPlaybackId PlaybackId, CriAtomPlayerHn Player)
{
	/* プレイリストの有無をチェック */
	FAtomPlaylist* Playlist = reinterpret_cast<FAtomPlaylist*>(Object);
	if (Playlist == nullptr) {
		return;
	}

	/* 排他制御区間の開始 */
	Playlist->Lock();

	/* 再生位置の取得 */
	int32 Pos = Playlist->Tell();

	/* リストアイテムの取得 */
	const FAtomPlaylistItem* Item = Playlist->Get(Pos);
	if ((Item != nullptr) && (Item->AwbHn != nullptr)) {
		/* 連結する波形データをセット */
		criAtomPlayer_SetWaveId(Player, Item->AwbHn, Item->WaveId);

		/* 再生位置の更新 */
		Playlist->Seek(Pos + 1);
	}

	/* 排他制御区間の終了 */
	Playlist->Unlock();
}

void UAtomComponent::Stop()
{
	/* 発音中かどうかチェック */
	if (bIsActive == false) {
		return;
	}

	/* 再生を停止 */
	/* 注意）GC処理の順序によっては、ライブラリ終了処理後にここに来る可能性がある。 */
	if (criAtomEx_IsInitialized() != CRI_FALSE) {
		if (Player != NULL) {
			criAtomExPlayer_Stop(Player);
		}
	}

	/* 再生の終了を通知 */
	bIsActive = false;
}

/* フェードイン */
void UAtomComponent::FadeIn(float FadeInDuration, float FadeVolumeLevel, float StartTime)
{
	PlayInternal(StartTime, FadeInDuration, FadeVolumeLevel);
}

/* フェードアウトして停止 */
void UAtomComponent::FadeOut(float FadeOutDuration, float FadeVolumeLevel)
{
	if (Player == NULL) {
		return;
	}

	if (FadeOutDuration > 0.0f) {
		float ReleaseTime = FMath::Min(FadeOutDuration * 1000.0f, 10000.0f);
		criAtomExPlayer_SetEnvelopeReleaseTime(Player, ReleaseTime);
		criAtomExPlayer_UpdateAll(Player);
	}

	Stop();
}

EAtomComponentStatus UAtomComponent::GetStatus()
{
	if (Player == NULL) {
		return EAtomComponentStatus::Error;
	}

	/* AtomExプレーヤのステータスを取得 */
	CriAtomExPlayerStatus PlayerStatus = criAtomExPlayer_GetStatus(Player);

	/* コンポーネントのステータスに変換 */
	EAtomComponentStatus ComponentStatus;
	switch (PlayerStatus) {
		case CRIATOMEXPLAYER_STATUS_STOP:
			ComponentStatus = EAtomComponentStatus::Stop;
			break;

		case CRIATOMEXPLAYER_STATUS_PREP:
			ComponentStatus = EAtomComponentStatus::Prep;
			break;

		case CRIATOMEXPLAYER_STATUS_PLAYING:
			ComponentStatus = EAtomComponentStatus::Playing;
			break;

		case CRIATOMEXPLAYER_STATUS_PLAYEND:
			ComponentStatus = EAtomComponentStatus::PlayEnd;
			break;

		default:
			ComponentStatus = EAtomComponentStatus::Error;
			break;
	}

	return ComponentStatus;
}

void UAtomComponent::Pause(bool bPause)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	if (bPause) {
		criAtomExPlayer_Pause(TargetPlayer, CRI_ON);
	} else {
		criAtomExPlayer_Resume(TargetPlayer, CRIATOMEX_RESUME_PAUSED_PLAYBACK);
	}
}

bool UAtomComponent::IsPaused()
{
	if (Player == NULL) {
		return false;
	}

	if (criAtomExPlayer_IsPaused(Player) == CRI_TRUE) {
		return true;
	} else {
		return false;
	}
}

bool UAtomComponent::IsPlaying(void)
{
	/* ステータスの確認 */
	EAtomComponentStatus Status = GetStatus();
	if ((Status == EAtomComponentStatus::Prep) || (Status == EAtomComponentStatus::Playing)) {
		return true;
	} else {
		return false;
	}
}

/* ボリュームの変更 */
void UAtomComponent::SetVolume(float Volume)
{
	/* ボリューム値の保存 */
	DefaultVolume = Volume;

	if (Player == NULL) {
		return;
	}

	/* 距離減衰の上書き判定 */
	if (((bOverrideAttenuation != false) || (AttenuationSettings != NULL) || (Sound && Sound->DefaultAttenuation)) && (bEnableDefaultAttenuation == true)) {
		/* 減衰ボリュームの計算 */
		ApplyAttenuation(Volume, Filter_Frequency);

		/* LPFの適用 */
		const auto* Settings = GetAttenuationSettingsToApply();
		criAtomExPlayer_SetBandpassFilterParameters(Player, 0.f, Filter_Frequency / MAX_FILTER_FREQUENCY);
	}

	/* ボリュームの更新 */
	criAtomExPlayer_SetVolume(Player, Volume);
	criAtomExPlayer_UpdateAll(Player);
}

/* サウンドオブジェクトの設定 */
void UAtomComponent::SetSoundObject(UAtomSoundObject * SoundObject)
{
	/* サウンドオブジェクトの適用 */
	ApplySoundObject(SoundObject);

	/* デフォルト値の更新 */
	DefaultSoundObject = SoundObject;
}

/* サウンドオブジェクトの適用 */
void UAtomComponent::ApplySoundObject(UAtomSoundObject * SoundObject)
{
	/* サウンドオブジェクトの更新が必要かどうかチェック */
	if (SoundObject == AppliedSoundObject) {
		return;
	}

	/* 適用済みのサウンドオブジェクトを解除 */
	if (AppliedSoundObject != nullptr) {
		if (Player != NULL) {
			criAtomExSoundObject_DeletePlayer(AppliedSoundObject->GetSoundObjectHandle(), Player);
		}
		AppliedSoundObject = nullptr;
	}

	/* 即時適用が必要かどうかチェック */
	if ((SoundObject == nullptr) || (Player == NULL)) {
		return;
	}

	/* サウンドオブジェクトの設定 */
	criAtomExSoundObject_AddPlayer(SoundObject->GetSoundObjectHandle(), Player);

	/* 適用済みサウンドオブジェクトの保持 */
	AppliedSoundObject = SoundObject;
}

void UAtomComponent::SetPitchMultiplier(float NewPitchMultiplier)
{
	/* ピッチをセント単位に変更 */
	float Cent = 1200.0f * FMath::Log2(NewPitchMultiplier);

	/* ピッチの変更 */
	SetPitch(Cent);
}

/* ピッチの変更 */
void UAtomComponent::SetPitch(float Pitch)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	criAtomExPlayer_SetPitch(TargetPlayer, Pitch);
	criAtomExPlayer_UpdateAll(TargetPlayer);
}

/* AISACコントロール値の指定 */
void UAtomComponent::SetAisacByName(FString ControlName, float ControlValue)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	criAtomExPlayer_SetAisacControlByName(TargetPlayer, TCHAR_TO_UTF8(*ControlName), ControlValue);
	criAtomExPlayer_UpdateAll(TargetPlayer);
}

/* 非推奨関数 */
void UAtomComponent::SetBusSendLevel(int32 BusId, float Level)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	criAtomExPlayer_SetBusSendLevel(TargetPlayer, BusId, Level);
	criAtomExPlayer_UpdateAll(TargetPlayer);
}

/* バスセンドレベルの設定 */
void UAtomComponent::SetBusSendLevelByName(FString BusName, float Level)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	criAtomExPlayer_SetBusSendLevelByName(TargetPlayer, TCHAR_TO_UTF8(*BusName), Level);
	criAtomExPlayer_UpdateAll(TargetPlayer);
}

/* 非推奨関数 */
void UAtomComponent::SetBusSendLevelOffset(int32 BusId, float LevelOffset)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	criAtomExPlayer_SetBusSendLevelOffset(TargetPlayer, BusId, LevelOffset);
	criAtomExPlayer_UpdateAll(TargetPlayer);
}

/* バスセンドレベルの設定（オフセット指定） */
void UAtomComponent::SetBusSendLevelOffsetByName(FString BusName, float LevelOffset)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	criAtomExPlayer_SetBusSendLevelOffsetByName(TargetPlayer, TCHAR_TO_UTF8(*BusName), LevelOffset);
	criAtomExPlayer_UpdateAll(TargetPlayer);
}

/* ブロックの遷移 */
void UAtomComponent::SetNextBlockIndex(int32 BlockIndex)
{
	DefaultBlockIndex = BlockIndex;
	criAtomExPlayback_SetNextBlockIndex(PlaybackId, BlockIndex);
}

/* セレクタラベルの設定 */
void UAtomComponent::SetSelectorLabel(FString Selector, FString Label)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	criAtomExPlayer_SetSelectorLabel(TargetPlayer, TCHAR_TO_UTF8(*Selector), TCHAR_TO_UTF8(*Label));
	criAtomExPlayer_UpdateAll(TargetPlayer);
}

/* ASRRackIDの設定 */
void UAtomComponent::SetAsrRackID(int32 asr_rack_id)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	criAtomExPlayer_SetAsrRackId(TargetPlayer, asr_rack_id);
}

void UAtomComponent::SetAsrRackIdArray(TArray<int32> AsrRackIDs)
{
	CriAtomExPlayerHn TargetPlayer = GetAtomExPlayer();
	if (TargetPlayer == NULL) {
		return;
	}

	int32 Index = 0;
	CriSint32 RackIdArray[CRIATOMEXPLAYER_MAX_ASR_RACKS];
	for (int32 RackId : AsrRackIDs) {
		RackIdArray[Index] = RackId;
		Index++;
	}
	criAtomExPlayer_SetAsrRackIdArray(TargetPlayer, RackIdArray, Index);
}

float UAtomComponent::GetTime()
{
	CriSint64 TimeMS = criAtomExPlayback_GetTime(PlaybackId);
	if (TimeMS < 0) {
		return -1.0f;
	} else {
		return ((float)TimeMS * 0.001f);
	}
}

float UAtomComponent::GetSequencePosition()
{
	CriSint64 TimeMS = criAtomExPlayback_GetSequencePosition(PlaybackId);
	if (TimeMS < 0) {
		return -1.0f;
	} else {
		return ((float)TimeMS * 0.001f);
	}
}

void UAtomComponent::DestroyComponentByID(int32 TargetID)
{
	/* IDに対応するAtomComponentを検索 */
	UAtomComponent* AtomComponent = GetAtomComponentFromID(TargetID);
	if (AtomComponent == nullptr) {
		return;
	}

	/* AtomComponentの破棄 */
	AtomComponent->DestroyComponent();
}

void UAtomComponent::SetDefaultAttenuationEnable(bool bEnable)
{
	bEnableDefaultAttenuation = bEnable;
}

bool UAtomComponent::GetDefaultAttenuationEnable()
{
	return bEnableDefaultAttenuation;
}

/* AtomExプレーヤハンドルの取得 */
CriAtomExPlayerHn UAtomComponent::GetAtomExPlayer()
{
	if (IsPendingKill() == true) {
		return NULL;
	}

	if (Player == NULL) {
		AllocateResource();
	}

	return Player;
}

/* AtomEx3dSourceハンドルの取得 */
CriAtomEx3dSourceHn UAtomComponent::GetAtomEx3dSource()
{
	if (IsPendingKill() == true) {
		return NULL;
	}

	if (Source == NULL) {
		AllocateResource();
	}

	return Source;
}

void UAtomComponent::PostInitProperties()
{
	Super::PostInitProperties();

	/* デフォルトオブジェクトかどうかチェック */
	if (HasAnyFlags(RF_ClassDefaultObject)) {
		/* デフォルトオブジェクトの場合は何もしない */
		return;
	}

	/* 注意）テンプレートクラス等で処理負荷が上がらないよう、	*/
	/* 　　　PostInitProperties時点ではPlayer等は確保しない。	*/
}

UAtomComponent* UAtomComponent::GetAtomComponentFromID(int32 TargetID)
{
	/* Mapに登録されているIDは1以上 */
	if (TargetID <= 0) {
		return nullptr;
	}

	check(IsInGameThread());
	return AtomIDToComponentMap.FindRef(TargetID);
}

void UAtomComponent::BeginDestroy()
{
	/* 再生の停止 */
	Stop();

	/* リソースの破棄 */
	ReleaseResource();

	/* デリゲートの登録解除 */
	if (OnWorldCleanedUpHandle.IsValid() != false) {
		FWorldDelegates::OnWorldCleanup.Remove(OnWorldCleanedUpHandle);
		OnWorldCleanedUpHandle.Reset();
	}

	/* ComponentをMapから削除 */
	if (!HasAnyFlags(RF_ClassDefaultObject)) {
		AtomIDToComponentMap.Remove(AtomComponentID);
	}

	Super::BeginDestroy();
}

void UAtomComponent::DestroyComponent(bool bPromoteChildren/*= false*/)
{
	/* 再生の停止 */
	Stop();

	/* リソースの破棄 */
	ReleaseResource();

	/* 再生中にコンポーネントが破棄される場合も通知 */
	NotifyAudioFinished();

	/* ルート化の解除 */
	/* 注意）この処理はSuper::DestroyComponent前に実行しておく必要がある。	*/
	/* 　　　（BeginDestroy内で行うとAssertに引っかかる。）					*/
	if (bPersistAcrossLevelTransition) {
		RemoveFromRoot();
		AtomRootedComponentArray.Remove(this);
		bPersistAcrossLevelTransition = false;
	}

	if (bAddedAtomNonRegisteredComponentArray) {
		AtomNonRegisteredComponentArray.Remove(this);
		bAddedAtomNonRegisteredComponentArray = false;
	}

	Super::DestroyComponent(bPromoteChildren);
}

#if WITH_EDITOR
void UAtomComponent::OnAttachmentChanged()
{
	Super::OnAttachmentChanged();

	if (GetAttachParent() == NULL) {
		/* 音源表示用コンポーネントの破棄 */
		if (SpriteComponent) {
			SpriteComponent->DestroyComponent();
			SpriteComponent = nullptr;
		}
		if (MaxDistanceSphereComponent) {
			MaxDistanceSphereComponent->DestroyComponent();
			MaxDistanceSphereComponent = nullptr;
		}
		if (MinDistanceSphereComponent) {
			MinDistanceSphereComponent->DestroyComponent();
			MinDistanceSphereComponent = nullptr;
		}
	}
}

void UAtomComponent::PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent)
{
	/* 再生をリスタート */
	Play();

#if WITH_EDITORONLY_DATA
	/* Auto Activateプロパティの変化に合わせて、スピーカーアイコンを変更 */
	UpdateSpriteTexture();
#endif

	Super::PostEditChangeProperty(PropertyChangedEvent);
}
#endif

void UAtomComponent::TickComponent(float DeltaTime, enum ELevelTick TickType, FActorComponentTickFunction *ThisTickFunction)
{
	Super::TickComponent(DeltaTime, TickType, ThisTickFunction);

	OnTickComponent(DeltaTime);
}

void UAtomComponent::OnTickComponent(float DeltaTime)
{
	/* 再生が開始されたかどうかチェック */
	if ((Player == nullptr) || ((bIsActive == false) && (bIsPlayed == false))) {
		/* 再生前は何もしない */
		return;
	}

	/* プレーヤのステータスをチェック */
	CriAtomExPlayerStatus Status = criAtomExPlayer_GetStatus(Player);
	if ((Status == CRIATOMEXPLAYER_STATUS_PREP) || (Status == CRIATOMEXPLAYER_STATUS_PLAYING)) {
		/* リスナの差し替え */
		FAtomListener* NewListener = FAtomListener::GetNearestListener(GetComponentLocation());
		if (NewListener != nullptr) {
			criAtomExPlayer_Set3dListenerHn(Player, NewListener->GetListenerHandle());
			criAtomExPlayer_UpdateAll(Player);
			Listener = NewListener;
		}

		/* 音源位置の更新 */
		if (Listener && bTransformUpdated) {
			UpdateTransform();
		}

		/* 音源速度の更新 */
		if (Listener && bIsEnableDoppler) {
			/* DopplerFactorが設定されている場合 */
			FVector CurrentSourceLocation = GetComponentLocation();
			FVector CurrentListenerLocation = Listener->GetListenerLocation();
			CriAtomExVector source_pos, listener_pos;
			source_pos.x = CurrentSourceLocation.X * DistanceFactor;
			source_pos.y = CurrentSourceLocation.Y * DistanceFactor;
			source_pos.z = CurrentSourceLocation.Z * DistanceFactor;
			listener_pos.x = CurrentListenerLocation.X * DistanceFactor;
			listener_pos.y = CurrentListenerLocation.Y * DistanceFactor;
			listener_pos.z = CurrentListenerLocation.Z * DistanceFactor;

			CriAtomExVector source_velocity, listener_velocity;
			source_velocity.x = (source_pos.x - PreviousSourceLocation.x) / DeltaTime;
			source_velocity.y = (source_pos.y - PreviousSourceLocation.y) / DeltaTime;
			source_velocity.z = (source_pos.z - PreviousSourceLocation.z) / DeltaTime;
			listener_velocity.x = (listener_pos.x - PreviousListenerLocation.x) / DeltaTime;
			listener_velocity.y = (listener_pos.y - PreviousListenerLocation.y) / DeltaTime;
			listener_velocity.z = (listener_pos.z - PreviousListenerLocation.z) / DeltaTime;

			CriAtomExVector relative_velocity;
			relative_velocity.x = source_velocity.x - listener_velocity.x;
			relative_velocity.y = source_velocity.y - listener_velocity.y;
			relative_velocity.z = source_velocity.z - listener_velocity.z;

			criAtomEx3dSource_SetVelocity(Source, &relative_velocity);

			PreviousSourceLocation.x = source_pos.x;
			PreviousSourceLocation.y = source_pos.y;
			PreviousSourceLocation.z = source_pos.z;
			PreviousListenerLocation.x = listener_pos.x;
			PreviousListenerLocation.y = listener_pos.y;
			PreviousListenerLocation.z = listener_pos.z;
			criAtomEx3dSource_Update(Source);
		}

		/* 距離減衰の上書き判定 */
		if ((((bOverrideAttenuation == true) || (AttenuationSettings != NULL) || (Sound && Sound->DefaultAttenuation)) && (bPersistAcrossLevelTransition == false)) && (bEnableDefaultAttenuation == true)) {
			/* 減衰ボリュームの計算 */
			float Volume = DefaultVolume;
			ApplyAttenuation(Volume, Filter_Frequency);

			/* ボリュームの更新 */
			criAtomExPlayer_SetVolume(Player, Volume);

			/* LPFの適用 */
			const auto* Settings = GetAttenuationSettingsToApply();
			criAtomExPlayer_SetBandpassFilterParameters(Player, 0.f, Filter_Frequency / MAX_FILTER_FREQUENCY);
			criAtomExPlayer_UpdateAll(Player);
		}

#if WITH_EDITOR
		/* Play In Editorモードでは、再生中に音源位置と減衰距離を表示する */
		if (GIsPlayInEditorWorld) {
			/* コンソール変数 "cri.ShowAtomSoundActor" が 0 より大きいときのみ表示する。 */
			static const auto bShowSoundLoation = IConsoleManager::Get().FindTConsoleVariableDataInt(TEXT("cri.ShowSoundLocation"));
			if (bShowSoundLoation->GetValueOnGameThread() > 0) {
				/* Play In Editorモードで表示するには、ActorとComponentのbHiddenGameプロパティを、両方とも false にする必要がある。 */
				if (GetOwner()) {
					GetOwner()->SetActorHiddenInGame(false);
				}
				SetHiddenInGame(false, true);

				/* Attenuationパラメータ設定時の描画処理 */
				DrawDebugShape();
			}
		}
#endif

		/* 再生済みのデータを破棄 */
		if ((Playlist != nullptr) && (Status == CRIATOMEXPLAYER_STATUS_PLAYING)) {
			/* 排他制御区間の開始 */
			Playlist->Lock();

			/* 備考）解放待ち、再生中、再生待ちの3パケットは保持しておく。 */
			int32 Pos = Playlist->Tell();
			if (Pos > 3) {
				Playlist->RemoveAt(0);
			}

			/* 排他制御区間の終了 */
			Playlist->Unlock();
		}

		return;
	}

	/* 再生の完了を通知 */
	NotifyAudioFinished();

	if (bAutoDestroy) {
		/* 再生の停止 */
		Stop();

		/* リソースの破棄 */
		/* 備考）GCまでの間ハンドルを保持し続けると、サーバ処理内でリストを     */
		/* 　　　辿る処理が無駄に発生し続けるので、ここでハンドルを破棄する。   */
		ReleaseResource();

		/* 再生終了時はコンポーネントを破棄 */
		DestroyComponent();

		/* 音源表示用コンポーネントの破棄 */
#if WITH_EDITOR
		if (SpriteComponent) {
			SpriteComponent->DestroyComponent();
			SpriteComponent = nullptr;
		}
		if (MaxDistanceSphereComponent) {
			MaxDistanceSphereComponent->DestroyComponent();
			MaxDistanceSphereComponent = nullptr;
		}
		if (MinDistanceSphereComponent) {
			MinDistanceSphereComponent->DestroyComponent();
			MinDistanceSphereComponent = nullptr;
		}
#endif
	} else {
#if WITH_EDITOR
		/* 再生終了したので、音源表示を隠す */
		if (GIsPlayInEditorWorld) {
			SetHiddenInGame(true, true);
		}
#endif
	}
}

void UAtomComponent::Activate(bool bReset)
{
	if (bReset || ShouldActivate() == true) {
		Play();
	}
}

void UAtomComponent::Deactivate()
{
	if (ShouldActivate() == false) {
		Stop();
	}
}

const FSoundAttenuationSettings* UAtomComponent::GetAttenuationSettingsToApply() const
{
	/* AttenuationSettingsを使用するかどうかチェック */
	if (bOverrideAttenuation != false) {
		/* AttenuationSettings上書き時 */
		return &AttenuationOverrides;
	} else if (AttenuationSettings != NULL) {
		/* AttenuationSettingsアセット使用時 */
		return &AttenuationSettings->Attenuation;
	} else if (bEnableDefaultAttenuation && Sound && Sound->DefaultAttenuation) {
		return &(Sound->DefaultAttenuation->Attenuation);
	}

	return NULL;
}

void UAtomComponent::ApplyAttenuation(float& Volume, float& FilterFrequency)
{
	AtomActiveSound.SetAtomComponent(this->GetOwner(), this->GetAtomComponentID(), this->GetFName());
	AtomActiveSound.SetWorld(GetWorld());

	/* AttenuationSettingsの取得 */
	const auto* Settings = GetAttenuationSettingsToApply();
	if (Settings == NULL) {
		return;
	}

	/* リスニングポイントの座標を取得 */
	FVector ListenerLocation = ((Listener != nullptr) ? Listener->GetListeningPoint() : FVector::ZeroVector);

	/* ソースの向きと座標を取得 */
	const FRotator SourceRotation = GetComponentRotation();
	const FVector SourcePosition = GetComponentLocation();

	/* ボリュームの初期値を計算 */
	FTransform SoundTransform = FTransform(SourceRotation, SourcePosition);
	Volume = DefaultVolume;

	/* 距離減衰の計算 */
	/* 備考）UE4.10相当の処理を移植。 */
	if (Settings->bAttenuate) {
		switch (Settings->AttenuationShape) {
			case EAttenuationShape::Sphere:
			{
				const float Distance = FMath::Max(
					FVector::Dist(SoundTransform.GetTranslation(), ListenerLocation)
					- Settings->AttenuationShapeExtents.X, 0.f);
				Volume *= Settings->AttenuationEval(Distance, Settings->FalloffDistance, 1.0f);
				break;
			}

			case EAttenuationShape::Box:
				Volume *= Settings->AttenuationEvalBox(SoundTransform, ListenerLocation, 1.0f);
				break;

			case EAttenuationShape::Capsule:
				Volume *= Settings->AttenuationEvalCapsule(SoundTransform, ListenerLocation, 1.0f);
				break;

			case EAttenuationShape::Cone:
				Volume *= Settings->AttenuationEvalCone(SoundTransform, ListenerLocation, 1.0f);
				break;

			default:
				check(false);
		}
	}

	// Only do occlusion traces if the sound is audible
	if (Settings->bEnableOcclusion && Volume > 0.0f /*&& !AudioDevice->IsAudioDeviceMuted()*/) {
		AtomActiveSound.CheckOcclusion(ListenerLocation, GetComponentLocation(), Settings);
		FilterFrequency = AtomActiveSound.CurrentOcclusionFilterFrequency.GetValue();
		// Apply the volume attenuation due to occlusion (using the interpolating dynamic parameter)

		Volume *= AtomActiveSound.CurrentOcclusionVolumeAttenuation.GetValue();
	}
}

/* スピーカーアイコン表示関連 */
#if WITH_EDITORONLY_DATA
/* Auto Activeならばチェックボックス付きのテクスチャを使う */
void UAtomComponent::UpdateSpriteTexture()
{
	if (SpriteComponent) {
		if (bAutoActivate) {
			SpriteComponent->SetSprite(LoadObject<UTexture2D>(NULL, TEXT("/Engine/EditorResources/AudioIcons/S_AudioComponent_AutoActivate.S_AudioComponent_AutoActivate")));
		} else {
			SpriteComponent->SetSprite(LoadObject<UTexture2D>(NULL, TEXT("/Engine/EditorResources/AudioIcons/S_AudioComponent.S_AudioComponent")));
		}
	}
}

void UAtomComponent::RegisterAtomSoundInfoComponent()
{
	/* エディターのパースペクティブビューに、スピーカーのアイコンを表示
	* インゲームでも、デバッグ表示向けにSpriteComponentを初期化しておく */
	/* SceneComponent.cpp USceneComponent::OnRegisterを参考 */
	if (bVisualizeComponent && SpriteComponent == nullptr && GetOwner()) {
		SpriteComponent = NewObject<UBillboardComponent>(GetOwner(), NAME_None, RF_Transactional | RF_TextExportTransient);

		SpriteComponent->Sprite = LoadObject<UTexture2D>(nullptr, TEXT("/Engine/EditorResources/EmptyActor.EmptyActor"));
		SpriteComponent->RelativeScale3D = FVector(0.5f, 0.5f, 0.5f);
		SpriteComponent->Mobility = EComponentMobility::Movable;
		SpriteComponent->AlwaysLoadOnClient = false;
		SpriteComponent->AlwaysLoadOnServer = false;
		SpriteComponent->SpriteInfo.Category = TEXT("Misc");
		SpriteComponent->SpriteInfo.DisplayName = NSLOCTEXT("SpriteCategory", "Misc", "Misc");
		SpriteComponent->CreationMethod = CreationMethod;
		SpriteComponent->bIsScreenSizeScaled = true;
		SpriteComponent->bUseInEditorScaling = true;

		SpriteComponent->AttachToComponent(this, FAttachmentTransformRules::KeepRelativeTransform);
		SpriteComponent->RegisterComponent();
	}
	UpdateSpriteTexture();

	/* Play In Editorモードでは、再生中に最大、最小減衰距離を表示する。 */
	if (GIsPlayInEditorWorld && (MaxDistanceSphereComponent == NULL) && GetOwner()
		&& (bOverrideAttenuation == false) && (AttenuationSettings == NULL)) {
		if (MaxDistanceSphereComponent == NULL) {
			MaxDistanceSphereComponent = NewObject<UDrawSphereComponent>(GetOwner(), NAME_None, RF_Transactional | RF_TextExportTransient);

			MaxDistanceSphereComponent->AttachToComponent(this, FAttachmentTransformRules::KeepRelativeTransform);
			MaxDistanceSphereComponent->AlwaysLoadOnClient = false;
			MaxDistanceSphereComponent->AlwaysLoadOnServer = false;
			MaxDistanceSphereComponent->CreationMethod = CreationMethod;

			/* 最大減衰距離をキューから情報取得して、描画コンポーネントに設定 */
			if (Sound) {
				auto MaxAttenuationDistance = Sound->GetMaxAttenuationDistance();
				auto SphereRadius = MaxAttenuationDistance / UCriWareInitializer::GetDistanceFactor();
				MaxDistanceSphereComponent->InitSphereRadius(SphereRadius);
			}

			MaxDistanceSphereComponent->SetCanEverAffectNavigation(false);
			MaxDistanceSphereComponent->RegisterComponent();
		}
		if (MinDistanceSphereComponent == NULL) {
			MinDistanceSphereComponent = NewObject<UDrawSphereComponent>(GetOwner(), NAME_None, RF_Transactional | RF_TextExportTransient);

			MinDistanceSphereComponent->AttachToComponent(this, FAttachmentTransformRules::KeepRelativeTransform);
			MinDistanceSphereComponent->AlwaysLoadOnClient = false;
			MinDistanceSphereComponent->AlwaysLoadOnServer = false;
			MinDistanceSphereComponent->CreationMethod = CreationMethod;

			/* 最小減衰距離をキューから情報取得して、描画コンポーネントに設定 */
			if (Sound) {
				auto MinAttenuationDistance = Sound->GetMinAttenuationDistance();
				auto SphereRadius = MinAttenuationDistance / UCriWareInitializer::GetDistanceFactor();
				MinDistanceSphereComponent->InitSphereRadius(SphereRadius);
			}

			MinDistanceSphereComponent->SetCanEverAffectNavigation(false);
			MinDistanceSphereComponent->RegisterComponent();
		}
	}
}

void UAtomComponent::OnRegister()
{
	Super::OnRegister();
	RegisterAtomSoundInfoComponent();
}
#endif /* WITH_EDITORONLY_DATA */

void UAtomComponent::OnUnregister()
{
	Super::OnUnregister();

	AActor* Owner = GetOwner();
	if (!Owner || bStopWhenOwnerDestroyed) {
		Stop();
	}
}

void UAtomComponent::OnUpdateTransform(EUpdateTransformFlags UpdateTransformFlags, ETeleportType Teleport)
{
	Super::OnUpdateTransform(UpdateTransformFlags, Teleport);

	/* 音源位置の更新を通知 */
	bTransformUpdated = true;
}

void UAtomComponent::UpdateTransform(bool bHasOwner)
{
	if (Source == NULL) {
		return;
	}

	AActor* Owner = GetOwner();

	/* ソースの座標を更新 */
	const FVector SourcePosition = (bHasOwner && Owner) ? Owner->GetActorLocation() : GetComponentLocation();
	CriAtomExVector pos;
	pos.x = SourcePosition.X * DistanceFactor;
	pos.y = SourcePosition.Y * DistanceFactor;
	pos.z = SourcePosition.Z * DistanceFactor;
	criAtomEx3dSource_SetPosition(Source, &pos);

	/* ソースの向きを更新 */
	const FRotator SourceRotation = (bHasOwner && Owner) ? Owner->GetActorRotation() : GetComponentRotation();
	CriAtomExVector front, top;
	FQuat SourceRot_Quat = SourceRotation.Quaternion();
	FVector FrontVector = SourceRot_Quat.GetForwardVector();
	FVector TopVector = SourceRot_Quat.GetUpVector();
	front.x = FrontVector.X;
	front.y = FrontVector.Y;
	front.z = FrontVector.Z;
	top.x = TopVector.X;
	top.y = TopVector.Y;
	top.z = TopVector.Z;
	criAtomEx3dSource_SetOrientation(Source, &front, &top);

	/* 更新の適用 */
	criAtomEx3dSource_Update(Source);

	/* 音源位置の更新完了を通知 */
	bTransformUpdated = false;
};

const UAtomSoundObject * UAtomComponent::GetAppliedSoundObject() const
{
	return AppliedSoundObject;
}

void UAtomComponent::AllocateResource()
{
	/* デフォルトオブジェクトかどうかチェック */
	if (HasAnyFlags(RF_ClassDefaultObject)) {
		/* デフォルトオブジェクトの場合は何もしない */
		return;
	}

	/* プレーヤハンドルが確保済みの場合は何もしない */
	if (Player != NULL) {
		return;
	}

	/* デフォルトのリスナを取得 */
	/* 注意）デフォルトで何らかのリスナを設定しておかないと	*/
	/* 　　　3Dポジショニング音再生時にエラーが発生する。	*/
	Listener = FAtomListener::GetListener();
	if (Listener == NULL) {
		return;
	}

	/* Listenr位置情報、Source位置情報の初期化 */
	FVector SourceLocation = GetComponentLocation();
	PreviousSourceLocation.x = SourceLocation.X * DistanceFactor;
	PreviousSourceLocation.y = SourceLocation.Y * DistanceFactor;
	PreviousSourceLocation.z = SourceLocation.Z * DistanceFactor;
	FVector ListenerLocation = Listener->GetListenerLocation();
	PreviousListenerLocation.x = ListenerLocation.X * DistanceFactor;
	PreviousListenerLocation.y = ListenerLocation.Y * DistanceFactor;
	PreviousListenerLocation.z = ListenerLocation.Z * DistanceFactor;

	/* ソースの作成 */
	Source = criAtomEx3dSource_Create(NULL, SourceWork, sizeof(SourceWork));
	if (Source == NULL) {
		/* ソースが確保できない場合は何もしない */
		return;
	}

	/* プレーヤの作成 */
	Player = criAtomExPlayer_Create(NULL, PlayerWork, sizeof(PlayerWork));
	if (Player == NULL) {
		/* プレーヤが確保できない場合はソースも破棄 */
		criAtomEx3dSource_Destroy(Source);
		Source = NULL;
		return;
	}

	if (bUsePlaylist != false) {
		/* シームレス連結再生用にプレイリストを作成 */
		if (Playlist == nullptr) {
			Playlist = new FAtomPlaylist();
		}

		/* データ要求コールバックを設定 */
		criAtomExPlayer_SetDataRequestCallback(Player, UAtomComponent::OnDataRequest, Playlist);
	}

	/* 距離係数の取得 */
	DistanceFactor = UCriWareInitializer::GetDistanceFactor();

	/* ソース、リスナをプレーヤに関連付け */
	criAtomExPlayer_Set3dListenerHn(Player, Listener->GetListenerHandle());
	criAtomExPlayer_Set3dSourceHn(Player, Source);

#if defined(XPT_TGT_IOS) || defined(XPT_TGT_ANDROID) || defined(XPT_TGT_SWITCH)
	/* 5.1ch環境向けにパンスピーカータイプを設定 */
	criAtomExPlayer_SetPanSpeakerType(Player, CRIATOMEX_PAN_SPEAKER_TYPE_4CH);
#else
	/* 7.1ch環境向けにパンスピーカータイプを設定 */
	criAtomExPlayer_SetPanSpeakerType(Player, CRIATOMEX_PAN_SPEAKER_TYPE_6CH);
#endif

	/* サウンドオブジェクトを実際に設定 */
	ApplySoundObject(DefaultSoundObject);
}

void UAtomComponent::ReleaseResource()
{
	/* デフォルトオブジェクトかどうかチェック */
	if (HasAnyFlags(RF_ClassDefaultObject)) {
		/* デフォルトオブジェクトの場合は何もしない */
		return;
	}

	/* キューシートのリリース */
	if (CueSheet != NULL) {
		CueSheet->Release();
		CueSheet = NULL;
	}

	/* サウンドの参照を破棄 */
	Sound = nullptr;

	/* ソースとプレーヤを破棄 */
	/* 注意）GC処理の順序によっては、ライブラリ終了処理後にここに来る可能性がある。 */
	if (criAtomEx_IsInitialized() != CRI_FALSE) {
		/* 補足）SoundObjectの適用を解除する必要なし。				*/
		/* 　　　criAtomExPlayer_Destroy関数内で登録が解除される。	*/
		if (Player != NULL) {
			criAtomExPlayer_Destroy(Player);
		}
		if (Source != NULL) {
			criAtomEx3dSource_Destroy(Source);
		}
	}

	/* パラメータのクリア */
	Source = NULL;
	Player = NULL;

	/* シームレス連結再生用のプレイリストを破棄 */
	if (Playlist != nullptr) {
		delete Playlist;
		Playlist = nullptr;
	}
}

#if WITH_EDITOR
void UAtomComponent::DrawDebugShape()
{
	/* AttenuationSettingsの取得 */
	const auto* Attenuation = GetAttenuationSettingsToApply();
	if (Attenuation == NULL) {
		return;
	}

	/* 減衰範囲の描画に必要なUWorldを取得 */
	UWorld* World = (GEditor ? (GEditor->PlayWorld) : nullptr);
	ULocalPlayer* LocalPlayer = (GEngine ? (GEngine->GetDebugLocalPlayer()) : nullptr);
	if (LocalPlayer) {
		UWorld* PlayerWorld = LocalPlayer->GetWorld();
		if (!World) {
			World = PlayerWorld;
		}
	}
	if (!World) {
		/* 取得失敗時は何もしない */
		return;
	}

	/* 距離減衰描画に必要なパラメータを取得 */
	const FRotator SourceRotation = GetComponentRotation();
	const FVector SourcePosition = GetComponentLocation();
	const FTransform SourceTransform = FTransform(SourceRotation, SourcePosition);

	/* 距離減衰形状の描画 */
	/* 備考）UnrealEngine.cppのUEngine::RenderStatSoundsを参考に作成。 */
	switch (Attenuation->AttenuationShape) {
		case EAttenuationShape::Sphere:
			if (Attenuation->FalloffDistance > 0.0f) {
				DrawDebugSphere(World, SourceTransform.GetTranslation(), Attenuation->AttenuationShapeExtents.X + Attenuation->FalloffDistance, 10, FColor(255, 0, 0));
				DrawDebugSphere(World, SourceTransform.GetTranslation(), Attenuation->AttenuationShapeExtents.X, 10, FColor(155, 0, 0));
			} else {
				DrawDebugSphere(World, SourceTransform.GetTranslation(), Attenuation->AttenuationShapeExtents.X, 10, FColor(255, 0, 0));
			}
			break;

		case EAttenuationShape::Box:
			if (Attenuation->FalloffDistance > 0.0f) {
				DrawDebugBox(World, SourceTransform.GetTranslation(), Attenuation->AttenuationShapeExtents + FVector(Attenuation->FalloffDistance), SourceTransform.GetRotation(), FColor(255, 0, 0));
				DrawDebugBox(World, SourceTransform.GetTranslation(), Attenuation->AttenuationShapeExtents, SourceTransform.GetRotation(), FColor(155, 0, 0));
			} else {
				DrawDebugBox(World, SourceTransform.GetTranslation(), Attenuation->AttenuationShapeExtents, SourceTransform.GetRotation(), FColor(255, 0, 0));
			}
			break;

		case EAttenuationShape::Capsule:
			if (Attenuation->FalloffDistance > 0.0f) {
				DrawDebugCapsule(World, SourceTransform.GetTranslation(), Attenuation->AttenuationShapeExtents.X + Attenuation->FalloffDistance, Attenuation->AttenuationShapeExtents.Y + Attenuation->FalloffDistance, SourceTransform.GetRotation(), FColor(255, 0, 0));
				DrawDebugCapsule(World, SourceTransform.GetTranslation(), Attenuation->AttenuationShapeExtents.X, Attenuation->AttenuationShapeExtents.Y, SourceTransform.GetRotation(), FColor(155, 0, 0));
			} else {
				DrawDebugCapsule(World, SourceTransform.GetTranslation(), Attenuation->AttenuationShapeExtents.X, Attenuation->AttenuationShapeExtents.Y, SourceTransform.GetRotation(), FColor(255, 0, 0));
			}
			break;

		case EAttenuationShape::Cone:
		{
			const FVector Origin = SourceTransform.GetTranslation() - (SourceTransform.GetUnitAxis(EAxis::X) * Attenuation->ConeOffset);

			if (Attenuation->FalloffDistance > 0.0f || Attenuation->AttenuationShapeExtents.Z > 0.0f) {
				const float OuterAngle = FMath::DegreesToRadians(Attenuation->AttenuationShapeExtents.Y + Attenuation->AttenuationShapeExtents.Z);
				const float InnerAngle = FMath::DegreesToRadians(Attenuation->AttenuationShapeExtents.Y);
				DrawDebugCone(World, Origin, SourceTransform.GetUnitAxis(EAxis::X), Attenuation->AttenuationShapeExtents.X + Attenuation->FalloffDistance + Attenuation->ConeOffset, OuterAngle, OuterAngle, 10, FColor(255, 0, 0));
				DrawDebugCone(World, Origin, SourceTransform.GetUnitAxis(EAxis::X), Attenuation->AttenuationShapeExtents.X + Attenuation->ConeOffset, InnerAngle, InnerAngle, 10, FColor(155, 0, 0));
			} else {
				const float Angle = FMath::DegreesToRadians(Attenuation->AttenuationShapeExtents.Y);
				DrawDebugCone(World, Origin, SourceTransform.GetUnitAxis(EAxis::X), Attenuation->AttenuationShapeExtents.X + Attenuation->ConeOffset, Angle, Angle, 10, FColor(255, 0, 0));
			}
		}
		break;

		default:
			check(false);
			break;
	}
}
#endif

void UAtomComponent::NotifyAudioFinished()
{
	/* 再生開始前は何もしない */
	if (bIsPlayed == false) {
		return;
	}

	/* イベントのトリガ */
	if (OnAudioFinished.IsBound()) {
		OnAudioFinished.Broadcast();
	}
	if (OnAudioFinishedNative.IsBound()) {
		OnAudioFinishedNative.Broadcast(this);
	}

	/* 再生終了を通知 */
	bIsPlayed = false;
}

void UAtomComponent::OnWorldCleanedUp(UWorld* World, bool bSessionEnded, bool bCleanupResources)
{
	/* 自身が所属するワールドかどうかチェック */
	UWorld* MyWorld = GetWorld();
	if (MyWorld != World) {
		return;
	}

	/* セッション終了時は音声を停止 */
	if (bSessionEnded || bCleanupResources) {
		Stop();
	}

	/* リソース破棄要求時はコンポーネントの破棄を要求 */
	if (bCleanupResources) {
		/* ワールド破棄時はイベントのトリガを禁止 */
		bIsPlayed = false;

		/* コンポーネントの破棄 */
		DestroyComponent();
	}
}

/***************************************************************************
 *      関数定義
 *      Function Definition
 ***************************************************************************/
#undef LOCTEXT_NAMESPACE

/* --- end of file --- */
