# CRI Middleware SDK
# Copyright (c) 2020 CRI Middleware Co., Ltd.
# CRI Atom Craft Robot Samples
# Version : 1.0.0
# 
# --Description:[ja]スイッチキューをセレクターから作成
# --Description:[en]Create a Switch Cue from Selector.
# --Description:[zh]从选择器创建切换Cue

import sys
import os
import cri.atomcraft.project
import cri.atomcraft.project as acproject
import cri.atomcraft.debug as acdebug

# --BeginUserVariable
"""
SELECTOR:
  type: object
  comment: The Selector to assign to the Cue
CUE_NAME:
  type: string
  comment: The name of the Cue to create
MATERIAL_FOLDER:
  type: object
  comment: (OPTION) Folder containing the Materials for the Cue
"""
SELECTOR = None
CUE_NAME = ""
MATERIAL_FOLDER = None
# --EndUserVariable

# Get the type of the selected object.
def get_first_object_from_selected(object_types):
    if type(object_types) is str:
        # Singular type specification
        result = acproject.get_selected_objects(object_types)
        if result["data"]:
            return result["data"][0]
    else:
        # Multiple type specifications
        for object_type in object_types:
            result = acproject.get_selected_objects(object_type)
            if result["data"]:
                return result["data"][0]
    return None

if not SELECTOR:
    acdebug.warning("Please specify a selector.")
    sys.exit()

if not CUE_NAME:
    acdebug.warning("Please specify the Cue name")
    sys.exit()

make_waveform_region = True
if not MATERIAL_FOLDER:
    make_waveform_region = False
    acdebug.log("Unable to register Waveform-regions because the Materials folder is not specified.")

# Get the labels from the Selector
selector_labels = acproject.get_child_objects(SELECTOR, "SelectorLabel")["data"]
if not selector_labels:
    acdebug.warning("Label not found.")
    sys.exit()

# Get the object to create the Cue into.
where_create_cue = get_first_object_from_selected(["CueSheet", "CueFolder"])

if not where_create_cue:
    acdebug.warning("Please select the CueSheet or Cue folder to create the Cue into.")
    sys.exit()

selector_name = acproject.get_value(SELECTOR, "Name")["data"]
acdebug.log("Start creating a Switch Cue \"{0}\" by specifying a Selector \"{0}\".".format(CUE_NAME, selector_name))

# Create the Cue
cue = acproject.create_object(where_create_cue, "Cue", CUE_NAME)["data"]

if not cue:
    acdebug.warning("Could not create the Cue.")
    sys.exit()

# Assign the Selector to the switch variable
acproject.set_values(cue, {"SequenceType": "Switch", "SwitchVariable": SELECTOR})

material_name_obj_pairs = []
if make_waveform_region:
    materials = acproject.get_child_objects(MATERIAL_FOLDER, "Material")["data"]
    for material in materials:
        name = acproject.get_value(material, "Name")["data"]
        material_name_obj_pairs.append([name, material])

# Create Tracks using the Selector label names
for label in selector_labels:
    label_name = acproject.get_value(label, "Name")["data"]
    track = acproject.create_object(cue, "Track", label_name)["data"]
    acproject.set_value(track, "SelectorLabel", label)
    # Register any Materials that partially match the Selector label name
    for material_name_obj_pair in material_name_obj_pairs:
        if label_name in material_name_obj_pair[0]:
            acproject.create_waveform_region(track, material_name_obj_pair[1])
            material_name_obj_pairs.remove(material_name_obj_pair)
            break

acdebug.log("The creation of the Switch Cue is completed.")