# CRI Middleware SDK
# Copyright (c) 2021 CRI Middleware Co., Ltd.
# CRI Atom Craft Robot Samples
# Version : 1.0.0
#
# --Description:[ja]仲間はずれのマテリアルを検索
# --Description:[en]Search for odd Materials
# --Description:[zh]检索相异的素材

import sys
import cri.atomcraft.debug as acdebug
import cri.atomcraft.project as acproject
import cri.atomcraft.project

# --BeginUserVariable
"""
BORDER_RATIO:
  type:number
  comment:The ratio to determine the oddness (0.0 < BORDER_RATIO < 0.5)
"""
BORDER_RATIO = 0.2
# --EndUserVariable

if BORDER_RATIO <= 0.0 or BORDER_RATIO >= 0.5:
    acdebug.warning("For BORDER_RATIO, please specify a value between 0.0 and 0.5.")
    sys.exit()

# Get the Material root folder.
material_folders = acproject.get_selected_objects("MaterialRootFolder")["data"]

# Get the sub-folder if the selected object is not the Material root folder.
if not material_folders:
    material_folders = acproject.get_selected_objects("MaterialSubFolder")["data"]
    if (len(material_folders) != 1):
        acdebug.warning("Please select the Material root folder or a sub-folder.")
        sys.exit()
        
material_folder = material_folders[0]
acdebug.log("Start searching for odd Materials.")

# List of fields to be investigated
field_names = [
    "SamplingRate",     # Sampling frequency
    "Channels",         # The number of channels
    "BitsPerSamples",   # Bit depth
    "LoopType",         # Loop information
    "EncodeType",       # Codec
    "EncodeQuality",    # HCA quality
    "StreamType",       # Stream type
]

# Get all the Materials in the selected object (including those in sub-folders).
materials = acproject.find_objects(material_folder, "Material")["data"]

# The function to find odd Materials using the information specified in the "field_name".
def search_different_materials(materials, field_name):
    field_value_dict = {}
    for material in materials:
        field_value = acproject.get_actual_value(material, field_name)["data"]

        # Store the Material's object name in the dictionary "field_value_dict".
        if not field_value_dict.get(field_value):
            # If field_value data is not contained in the dictionary,
            # register a new key and store the "material".
            field_value_dict[field_value] = [material]
        else:
            # If there is "field_value" data, store the "material".
            field_value_dict[field_value].append(material)
            
    # Check the odd "field_name"
    for field_value in field_value_dict:
        ratio = len(field_value_dict[field_value]) / len(materials)
        
        # If ratio is equal to or larger than BORDER_RATIO, do not continue.
        if ratio >= BORDER_RATIO:
            continue

        for material in field_value_dict[field_value]:
            material_name = acproject.get_value(material, "Name")["data"]
            acdebug.log(f"\"{material_name}\" has \"{field_name}\" of \"{field_value}\", which is different from the setting of most other Materials.")

# Check the fields set to the "field_names".
for field_name in field_names:
    search_different_materials(materials, field_name)

acdebug.log("Completed searching for odd Materials.")