﻿/****************************************************************************
 *
 * CRI Middleware SDK
 *
 * Copyright (c) 2021 CRI Middleware Co., Ltd.
 *
 * Library  : CRIWARE plugin for Unreal Engine
 * Module   : CriWareCore
 * File     : AtomResourceManager.h
 *
 ****************************************************************************/

#pragma once

#include "CoreMinimal.h"

#include "CriWareUtils.h"

#include "Atom/Atom.h"

// Forward Declarations
class IAtomSoundResource;
class FAtomResource;

/**
 * Typed identifier for Atom Resource Id
 */
using FAtomResourceId = uint32;

// this enum is used as result when a resource is loaded.  
enum class EAtomResourceLoadResult : uint8
{
	Completed,
	AlreadyLoaded,
	Interrupted,
	OutOfBounds,
	CacheBlown
};

// This enum is used in FAtomResourceManager::RequestResource to map a given Atom runtime to an Atom resource.
enum class EAtomResourceScope : uint8
{
	// Default to the behavior specified by the editor preferences.
	//Default = Shared,
	// Use an Atom resource that can be shared by multiple runtimes. (The resource can be loaded and unlaoded when it change.)
	Shared,
	// Create a new Atom resource specifically for the given associated object.
	Unique
};

// Parameters passed into FAtomResourceManager::RequestResource.
struct FAtomLoadParams
{
	// The source object used to load and create this resource.
	IAtomSoundResource* Source;
	// Optional object parameter. This allows to inform the object about resource loading or destruction if the object use IAtomResource interface. This also allows tools to surface information about which resources are being accessed through which objects.
	UObject* AssociatedObject;
	// This should be set to EAtomResourceScope::Unique if you'd like to force a new resource to be created from scratch, or use EAtomResourceScope::Shared to let new Atom runtime reuse an existing resource if possible.
	EAtomResourceScope Scope;
	// The main runtime used for this resource. If the runtime is active, the resource can be used.
	FAtomRuntimeId RuntimeID;
	// The way to read the resource data. Streamed mean the the data will be streamed from the file if non inlined. if resource is inlined, the resource cannot be streamed.
	bool bIsStreamed;
	// The way to load the resource data. if sync, it will not return the RequestResource function before loading is complete. 
	bool bAsyncLoad;

	TFunction<void(EAtomResourceLoadResult)> OnLoadCompleted;
	
	ENamedThreads::Type ThreadToCallOnLoadCompletedOn;

	FAtomLoadParams()
		: Source(nullptr)
		, AssociatedObject(nullptr)
		, Scope(EAtomResourceScope::Unique)
		, RuntimeID(INDEX_NONE)
		, bIsStreamed(false)
		, bAsyncLoad(false)
		, OnLoadCompleted(nullptr)
		, ThreadToCallOnLoadCompletedOn(ENamedThreads::GameThread)
	{}
};

// Strong handle to an Atom resource. Guarantees that the Atom resource it references will stay alive while it is in scope.
class CRIWARECORE_API FAtomResourceHandle
{
public:

	FAtomResourceHandle();
	FAtomResourceHandle(const FAtomResourceHandle& Other);
	FAtomResourceHandle(FAtomResourceHandle&& Other);

	FAtomResourceHandle& operator=(const FAtomResourceHandle& Other);
	FAtomResourceHandle& operator=(FAtomResourceHandle&& Other);

	~FAtomResourceHandle();

	// Gets a pointer to the Atom resource. If invalid, returns nullptr.
	FAtomResource* GetResource() const;

	// Returns the Atom ID for the Atom runtime referenced by this handle.
	FAtomResourceId GetResourceID() const;

	// Returns object set when handle was created
	TWeakObjectPtr<UObject> GetObject() const;

	// Checks whether this points to a valid Atom runtime.
	bool IsValid() const;

	// Resets handle to invalid state
	void Reset();

private:

	// This constructor should only be called by FAtomResourceManager.
	FAtomResourceHandle(FAtomResource* InResource, FAtomResourceId InResourceID, UObject* InObject);

	// The Object that this FAtomResourceHandle was requested with.
	// Null if this Resource handle wasn't generated by a RequestXXXResource function.
	TWeakObjectPtr<UObject> Object;

	FAtomResource* Resource;
	FAtomResourceId ResourceID;

	friend class FAtomResourceManager;

public:

	// These are convenience operators to use an FAtomResourceHandle like an FAtomResource*.
	// For safety, we still require explicit casting to an FAtomResource* to ensure ownership isn't lost due to programmer error.
	const FAtomResource& operator*() const
	{
		check(IsValid());
		return *Resource;
	}

	FAtomResource& operator*()
	{
		check(IsValid());
		return *Resource;
	}

	const FAtomResource* operator->() const
	{
		check(IsValid());
		return Resource;
	}

	FAtomResource* operator->()
	{
		check(IsValid());
		return Resource;
	}

	FORCEINLINE explicit operator bool() const
	{
		return IsValid();
	}

	FORCEINLINE bool operator==(const FAtomResourceHandle& Other) const
	{
		return ResourceID == Other.ResourceID;
	}

	FORCEINLINE bool operator==(const FAtomResource*& Other) const
	{
		return Resource == Other;
	}

	FORCEINLINE bool operator==(FAtomResource*& Other) const
	{
		return Resource == Other;
	}

	FORCEINLINE bool operator==(const FAtomResourceId& Other) const
	{
		return ResourceID == Other;
	}

	explicit operator FAtomResource* () const
	{
		check(IsValid());
		return Resource;
	}

	explicit operator FAtomResourceId() const
	{
		return ResourceID;
	}
};

// List of delegates for the Atom manager.
class CRIWARECORE_API FAtomResourceManagerDelegates
{
public:
	// This delegate is called whenever an entirely new Atom resource is created.
	DECLARE_MULTICAST_DELEGATE_OneParam(FOnAtomResourceCreated, FAtomResourceId /* AtomResourceID */);
	static FOnAtomResourceCreated OnAtomResourceCreated;

	// This delegate is called whenever an Atom resource is destroyed.
	DECLARE_MULTICAST_DELEGATE_OneParam(FOnAtomResourceDestroyed, FAtomResourceId /* AtomResourceID */);
	static FOnAtomResourceDestroyed OnAtomResourceDestroyed;
};

/*
 * AtomResourceManager class
 * 
 * Class to manage all resources loaded or bound into Atom runtime. 
 */
class CRIWARECORE_API FAtomResourceManager
{
public:

	FAtomResourceManager();
	~FAtomResourceManager();

	/** Request a new resorce or retrun an already created one. */
	FAtomResourceHandle RequestSoundResource(const FAtomLoadParams& Params);

	/**
	 * Returns whether the Atom resource handle is valid (i.e. points to an actual Atom resource instance).
	 */
	bool IsValidResource(FAtomResourceId ResourceID) const;

	/**
	 * Returns a strong handle to the Atom resource associated with the given Atom resource ID.
	 * if the Atom resource ID is invalid returns an invalid, zeroed handle.
	 */
	FAtomResourceHandle GetResource(FAtomResourceId InResourceID);

	/**
	 * Returns a raw ptr to the Atom resource associated with the handle
	 * If the handle is invalid then a nullptr will be returned.
	 */
	FAtomResource* GetResourceRaw(FAtomResourceId InResourceID);

	/** Update Atom resources. check if some should be unloaded. */
	void UpdateAtomResources(bool bGameTicking);

	/** Iterator for each resources. */
	void IterateOverAllResources(TUniqueFunction<void(FAtomResourceId, FAtomResource*)> ForEachResource);

	/** Iterator for each resources using the given runtime ID. */
	void IterateOverAllResources(FAtomRuntimeId InRuntimeID, TUniqueFunction<void(FAtomResourceId, FAtomResource*)> ForEachResource);

#if WITH_EDITOR 
	/** Iterator for each resources with some information about current usage. */
	void IterateOverAllResourcesForStatistics(TUniqueFunction<void(FAtomResourceId, FAtomResource*, int32 NumberOfHandles, EAtomResourceScope)> ForEachResource);
#endif

	/** Returns the current number of Atom resources. */
	uint32 GetNumAtomResources() const;

	/** Returns all the Atom resources managed by Atom resource manager. */
	TArray<FAtomResource*> GetAtomResources();

	/** Returns all resources using the given runtime ID. */
	TArray<FAtomResource*> GetAtomResources(FAtomRuntimeId InRuntimeID);

	/** Returns all the objects associated with the given Atom resource ID. */
	TArray<UObject*> GetObjectsUsingAtomResource(FAtomResourceId InResourceID);

	/** */
	EAtomResourceScope GetScopeOfAtomResource(FAtomResourceId InResourceID);

	/** Unload a resource if not already unloaded, then change its runtime ID. */
	bool MoveResourceToRuntime(FAtomResourceId InResourceID, FAtomRuntimeId RuntimeID);

	// Calls OnLoadCompleted on current thread if CallbackThread == ENamedThreads::AnyThread, and dispatchs an async task on a named thread otherwise.
	static void ExecuteOnLoadCompleteCallback(EAtomResourceLoadResult Result, const TFunction<void(EAtomResourceLoadResult)>& OnLoadCompleted, const ENamedThreads::Type& CallbackThread);

private:

	// Create a new Atom resource in the manager.
	FAtomResourceHandle CreateNewResource(const FAtomLoadParams& InParams);

	/** Creates a handle given the index and a generation value. */
	uint32 GetNewResourceID();

	// Called exclusively by the FAtomResourceHandle copy constructor and assignment operators.
	void IncrementResource(FAtomResourceId AtomResourceID);

	// Called exclusively by the FAtomRuntimeHandle dtor.
	void DecrementResource(FAtomResourceId AtomResourceID);

	struct FResourceContainer
	{
		// Singularly owned resource.
		TUniquePtr<FAtomResource> Resource;

		// Ref count of FAtomResourceHandles referencing this Atom resource.
		int32 NumberOfHandlesToThisResource;

		/** Optional Objects that have been registered to this Atom resource. */
		TArray<TObjectPtr<UObject>> ObjectsUsingThisResource;

		/** Whether this Atom resource can be shared accross runtimes. */
		EAtomResourceScope Scope;

		// todo make this a shared ptr proxy
		IAtomSoundResource* Source;

		// When scope for this resource is unique, only this runtime ID can load and use the resouce.
		FAtomRuntimeId RuntimeID;
		
		// Declare this resource streamed
		bool bIsStreamed;
		
		// if true, loading is asychrone
		bool bIsLoadAsync;

		FResourceContainer(const FAtomLoadParams& InParams, FAtomResourceId InResourceID, FAtomResourceManager* Manager);
		FResourceContainer(FResourceContainer&& Other);

		FResourceContainer(const FResourceContainer& Other) = delete;
		FResourceContainer() = delete;

		~FResourceContainer();

		void LoadResource(const FAtomLoadParams& InParams, FAtomResourceId InResourceID, FAtomResourceManager* Manager);
	};

	FAtomResourceHandle BuildNewHandle(FResourceContainer& InContainer, FAtomResourceId InResourceID, const FAtomLoadParams& InParams);

	/** Registers the world with the provided Atom ID. */
	void RegisterObject(UObject* InObject, FResourceContainer& InContainer, FAtomResourceId InResourceID);

	/** Unregisters the world from the provided Atom ID. */
	void UnregisterObject(UObject* InObject, FResourceContainer& InContainer, FAtomResourceId InResourceID);

	/**
	 * This function is used to check if we can use an existing Atom resource.
	 */
	static bool CanUseResource(const FAtomLoadParams& InParams, const FResourceContainer& InContainer);

	// Map that UAtomSoundBanks, FAtomSoundBankProxys, and FAtomResourceHandles can use to
	// quickly lookup where their resource is currently stored in the manager.
	TMap<FAtomResourceId, FResourceContainer> Resources;

	// Critical section: only used when we are modifying element positions in the cache. This only happens in TouchElement, EvictLeastRecentChunk, and TrimMemory.
	// Individual resources should be thread safe to access.
	mutable FCriticalSection ResourceMapCriticalSection;

	/* Counter used by GetNewResouceID() to generate a unique ID for a given Atom resource. */
	uint32 ResourceIDCounter;
	
	// Number of async load operations we have currently in flight.
	FThreadSafeCounter NumberOfLoadsInFlight;

	friend class FAtomResourceHandle;
};
