﻿/****************************************************************************
 *
 * CRI Middleware SDK
 *
 * Copyright (c) 2021 CRI Middleware Co., Ltd.
 *
 * Library  : CRIWARE plugin for Unreal Engine
 * Module   : CriWareCore
 * File     : AtomRuntimeManager.h
 *
 ****************************************************************************/

#pragma once

#include "CoreMinimal.h"
#include "HAL/CriticalSection.h"
#include "UObject/WeakObjectPtr.h"

#include "CriWareDefines.h"

// Forward declarations
class UWorld;
class UAtomComponent;
class UAtomSoundClass;
class IAtomSoundResource;
class FAtomRuntime;
class ICriWarePlatformAtom;
class FAtomResourceManager;
class FAtomMonitor;

//namespace Atom
//{
	//class FMixerDevice;

	/**
	 * Typed identifier for Atom Runtime Id
	 */
	using FAtomRuntimeId = uint32;
//}

namespace Atom
{
	class FAtomDebugger;
}

enum class EAtomSoundType : uint8
{
	Class,
	Cue,
	Wave
};

// This enum is used in FAtomRuntimeManager::RequestAtomRuntime to map a given UWorld to an Atom runtime.
enum class EAtomRuntimeScope : uint8
{
	// Default to the behavior specified by the editor preferences.
	Default,
	// Use an Atom runtime that can be shared by multiple worlds.
	Shared,
	// Create a new Atom runtime specifically for this handle.
	Unique
};

// Parameters passed into FAtomRuntimeManager::RequestAtom.
struct FAtomRuntimeParams
{
	// Optional world parameter. This allows tools to surface information about which worlds are being rendered through which audio devices.
	UWorld* AssociatedWorld;
	// This should be set to EAudioDeviceScope::Unique if you'd like to force a new device to be created from scratch, or use EAudioDeviceScope::Shared to use an existing device if possible.
	EAtomRuntimeScope Scope;
	// Set this to true to get a handle to a non realtime audio renderer.
	bool bIsNonRealtime;
	// Use this to force this Atom runtime to use a specific runtime.
	ICriWarePlatformAtom* AtomModule;

	FAtomRuntimeParams()
		: AssociatedWorld(nullptr)
		, Scope(EAtomRuntimeScope::Default)
		, bIsNonRealtime(false)
		, AtomModule(nullptr)
	{}
};

// Strong handle to an Atom runtime. Guarantees that the Atom runtime it references will stay alive while it is in scope.
class CRIWARECORE_API FAtomRuntimeHandle
{
public:

	FAtomRuntimeHandle();
	FAtomRuntimeHandle(const FAtomRuntimeHandle& Other);
	FAtomRuntimeHandle(FAtomRuntimeHandle&& Other);

	FAtomRuntimeHandle& operator=(const FAtomRuntimeHandle& Other);
	FAtomRuntimeHandle& operator=(FAtomRuntimeHandle&& Other);


	~FAtomRuntimeHandle();

	// Gets a pointer to the Atom runtime.
	FAtomRuntime* GetAtomRuntime() const;

	// Returns the Atom ID for the Atom runtime referenced by this handle.
	FAtomRuntimeId GetRuntimeID() const;

	// Returns world set when handle was created
	TWeakObjectPtr<UWorld> GetWorld() const;

	// Checks whether this points to a valid Atom runtime.
	bool IsValid() const;

	// Resets handle to invalid state
	void Reset();

private:

	// This constructor should only be called by FAtomRuntimeManager.
	FAtomRuntimeHandle(FAtomRuntime* InAtom, FAtomRuntimeId InID, UWorld* InWorld);

	// The world that this FAtomRuntimeHandle was requested with.
	// Null if this Atom handle wasn't generated by RequestAtom.
	TWeakObjectPtr<UWorld> World;

	FAtomRuntime* Runtime;
	FAtomRuntimeId RuntimeID;

/*#if INSTRUMENT_AUDIODEVICE_HANDLES
	uint32 StackWalkID;
	void AddStackDumpToAudioDeviceContainer();
#endif*/

	friend class FAtomRuntimeManager;

public:

	// These are convenience operators to use an FAudioDeviceHandle like an FAudioDevice* or an Audio::FDeviceId.
	// For safety, we still require explicit casting to an FAudioDevice* to ensure ownership isn't lost due to programmer error.
	const FAtomRuntime& operator*() const
	{
		check(IsValid());
		return *Runtime;
	}

	FAtomRuntime& operator*()
	{
		check(IsValid());
		return *Runtime;
	}

	const FAtomRuntime* operator->() const
	{
		check(IsValid());
		return Runtime;
	}

	FAtomRuntime* operator->()
	{
		check(IsValid());
		return Runtime;
	}

	FORCEINLINE explicit operator bool() const
	{
		return IsValid();
	}

	FORCEINLINE bool operator==(const FAtomRuntimeHandle& Other) const
	{
		return RuntimeID == Other.RuntimeID;
	}

	FORCEINLINE bool operator==(const FAtomRuntime*& Other) const
	{
		return Runtime == Other;
	}

	FORCEINLINE bool operator==(FAtomRuntime*& Other) const
	{
		return Runtime == Other;
	}

	FORCEINLINE bool operator==(const FAtomRuntimeId& Other) const
	{
		return RuntimeID == Other;
	}

	explicit operator FAtomRuntime* () const
	{
		check(IsValid());
		return Runtime;
	}

	explicit operator FAtomRuntimeId() const
	{
		return RuntimeID;
	}
};

// List of delegates for the Atom manager.
class CRIWARECORE_API FAtomRuntimeManagerDelegates
{
public:

	// This delegate is called whenever an entirely new Atom runtime is created.
	DECLARE_MULTICAST_DELEGATE_OneParam(FOnAtomRuntimeCreated, FAtomRuntimeId /* RuntimeID */);
	static FOnAtomRuntimeCreated OnAtomRuntimeCreated;

	// This delegate is called whenever an Atom runtime is destroyed.
	DECLARE_MULTICAST_DELEGATE_OneParam(FOnAtomRuntimeDestroyed, FAtomRuntimeId /* RuntimeID */);
	static FOnAtomRuntimeDestroyed OnAtomRuntimeDestroyed;

	// This delegate is called when an Atom runtime is about to be activated.
	DECLARE_MULTICAST_DELEGATE_OneParam(FOnAtomRuntimeWillActivate, FAtomRuntimeId /* RuntimeID */);
	static FOnAtomRuntimeWillActivate OnAtomRuntimeWillActivate;

	// This delegate is called when an Atom runtime is activated and starts rendering sound.
	DECLARE_MULTICAST_DELEGATE_OneParam(FOnAtomRuntimeActivated, FAtomRuntimeId /* RuntimeID */);
	static FOnAtomRuntimeActivated OnAtomRuntimeActivated;

	// This delegate is called when an Atom runtime is about to be deactivated.
	DECLARE_MULTICAST_DELEGATE_OneParam(FOnAtomRuntimeWillDeactivate, FAtomRuntimeId /* RuntimeID */);
	static FOnAtomRuntimeWillDeactivate OnAtomRuntimeWillDeactivate;

	// This delegate is called when an Atom runtime is deactivated.
	DECLARE_MULTICAST_DELEGATE_OneParam(FOnAtomRuntimeDeactivated, FAtomRuntimeId /* RuntimeID */);
	static FOnAtomRuntimeDeactivated OnAtomRuntimeDeactivated;
};

// List of delegates for the world being registered to an Atom runtime.
class CRIWARECORE_API FAtomRuntimeWorldDelegates
{
public:
	// Called whenever a world is registered to an Atom runtime. UWorlds are not guaranteed to be registered to the same
	// Atom runtime throughout their lifecycle, and there is no guarantee on the lifespan of both the UWorld and the Atom
	// runtime registered in this callback.
	DECLARE_MULTICAST_DELEGATE_TwoParams(FOnWorldRegisteredToAtomRuntime, const UWorld* /*InWorld */, FAtomRuntimeId /* AtomRuntimeId*/);
	static FOnWorldRegisteredToAtomRuntime OnWorldRegisteredToAtomRuntime;

	// Called whenever a world is unregistered from an Atom runtime. UWorlds are not guaranteed to be registered to the same
	// Atom runtime throughout their lifecycle, and there is no guarantee on the lifespan of both the UWorld and the Atom
	// runtime registered in this callback.
	DECLARE_MULTICAST_DELEGATE_TwoParams(FOnWorldUnregisteredWithAtomRuntime, const UWorld* /*InWorld */, FAtomRuntimeId /* AtomRuntimeId*/);
	static FOnWorldUnregisteredWithAtomRuntime OnWorldUnregisteredWithAtomRuntime;
};

 /*
  * RuntimeManager class
  * 
  * Hold multilple Atom class to switch between them. 
  */
class CRIWARECORE_API FAtomRuntimeManager
{
public:

	/**
	 * Constructor
	 */
	FAtomRuntimeManager();

	/**
	 * Destructor
	 */
	~FAtomRuntimeManager();

	/** Returns the handle to the main Atom runtime. */
	FAtomRuntimeHandle GetMainAtomRuntimeHandle() const { return MainAtomRuntimeHandle; }
	FAtomRuntime* GetMainAtomRuntimeRaw() const { return MainAtomRuntimeHandle.GetAtomRuntime(); }
	FAtomRuntimeId GetMainAtomRuntimeID() const { return MainAtomRuntimeHandle.GetRuntimeID(); }

	/** Get the associated runtime handle to a world. */
	static FAtomRuntimeHandle GetAtomRuntimeFromWorld(const UWorld* InWorld);

	/** Get the associated runtime raw pointer to a world. */
	static FAtomRuntime* GetAtomRuntimeRawFromWorld(const UWorld* InWorld);
	
	/** Get the associated runtime raw pointer to a world context. */
	static FAtomRuntime* GetAtomRuntimeFromWorldContext(const UObject* WorldContextObject);

	FAtomRuntimeParams GetDefaultParamsForNewWorld();

	/**
	 * Creates an Atom runtime instance internally and returns a handle to the Atom runtime.
	 * This Atom runtime is guaranteed to be alive as long as the returned handle is in scope.
	 */
	FAtomRuntimeHandle RequestAtomRuntime(const FAtomRuntimeParams& InParams);

	/**
	 * Returns whether the Atom runtime handle is valid (i.e. points to an actual runtime instance).
	 */
	bool IsValidAtomRuntime(FAtomRuntimeId RuntimeID) const;

	/**
	 * Returns a strong handle to the audio device associated with the given Atom ID.
	 * if the Atom ID is invalid returns an invalid, zeroed handle.
	 */
	FAtomRuntimeHandle GetAtomRuntime(FAtomRuntimeId InRuntimeID);

	/**
	 * Returns a raw ptr to the Atom runtime associated with the handle
	 * If the handle is invalid then a nullptr will be returned.
	 */
	FAtomRuntime* GetAtomRuntimeRaw(FAtomRuntimeId InRuntimeID);
	const FAtomRuntime* GetAtomRuntimeRaw(FAtomRuntimeId InRuntimeID) const;

	/**
	  Sets the Atom runtime associated with the given world.
	  */
	void SetAtomRuntimeWorld(UWorld& InWorld, FAtomRuntimeId InRuntimeID);

	/**
	 * Initialize the Atom manager.
	 * Return true if successfully initialized.
	 **/
	static bool Initialize();
	static FAtomRuntimeManager* Get();
	static void Shutdown();

	/** Creates the main Atom runtime. */
	bool CreateMainAtomRuntime();

	/**
	 * Returns a ptr to the active Atom runtime. If there is no active
	 * runtime then it will return the main Atom runtime.
	 */
	FAtomRuntimeHandle GetActiveAtomRuntime();

	/** Returns the current number of Atom runtimes. */
	uint8 GetNumAtomRuntimes() const;

	/** Returns the number of worlds (e.g. PIE viewports) using the main Atom runtime. */
	uint8 GetNumMainAtomRuntimeWorlds() const;

	/** Returns all the Atom runtimes managed by Atom manager. */
	TArray<FAtomRuntime*> GetAtomRuntimes();

	/** Returns all the worlds associated with the given Atom ID. */
	TArray<UWorld*> GetWorldsUsingAtomRuntime(const FAtomRuntimeId& InRuntimeID);

	/** Update Atom runtimes. */
	void UpdateAtomRuntimes(bool bGameTicking);

	/** Iterator for each Atom runtime. */
	void IterateOverAllRuntimes(TUniqueFunction<void(FAtomRuntimeId, FAtomRuntime*)> ForEachRuntime);
	void IterateOverAllRuntimes(TUniqueFunction<void(FAtomRuntimeId, const FAtomRuntime*)> ForEachRuntime) const;

#if WITH_EDITOR
	/** Iterator for each Atom runtime with some information about current usage. */
	void IterateOverAllRuntimesForStatistics(TUniqueFunction<void(FAtomRuntimeId, FAtomRuntime*, int32 NumberOfHandles, EAtomRuntimeScope)> ForEachRuntime);
#endif

	void LogListOfAtomRuntimes();

	/** Sets which Atom runtime is the active Atom runtime. */
	void SetActiveAtomRuntime(FAtomRuntimeId InRuntimeID);

	/** Gets a reference to the Atom resource Manager */
	FAtomResourceManager& GetAtomResourceManager() const;

	/** Tracks objects in the all Atom runtimes. */
	void AddReferencedObjects(FReferenceCollector& Collector);

	/** Stops sounds using the given resource in all Atom runtimes. */
	void StopSoundsUsingResource(IAtomSoundResource* InSound, TArray<UAtomComponent*>* StoppedComponents = nullptr);

	/** Frees the given sound resource from the resource manager */
	void FreeResource(IAtomSoundResource* InSound);

	/** Registers the sound class for all active runtimes. */
	void RegisterSoundClass(UAtomSoundClass* SoundClass);

	/** Unregisters the sound class for all active runtimes. */
	void UnregisterSoundClass(UAtomSoundClass* SoundClass);

	/** Initializes sound classes for all active runtimes. */
	void InitSoundClasses();

	/** Registers the world with the provided Atom ID. */
	void RegisterWorld(UWorld* InWorld, FAtomRuntimeId RuntimeID);

	/** Unregisters the world from the provided Atom ID. */
	void UnregisterWorld(UWorld* InWorld, FAtomRuntimeId RuntimeID);

	/** Is debug visualization of 3d sounds enabled */
	bool IsVisualizeDebug3dEnabled() const;

	/** Toggles 3d visualization of 3d sounds on/off */
	void ToggleVisualize3dDebug();

	/** Reset all sound cue trims */
	void ResetAllDynamicSoundVolumes();

	/** Get, reset, or set a sound cue trim */
	float GetDynamicSoundVolume(EAtomSoundType SoundType, const FName& SoundName) const;
	void ResetDynamicSoundVolume(EAtomSoundType SoundType, const FName& SoundName);
	void SetDynamicSoundVolume(EAtomSoundType SoundType, const FName& SoundName, float Volume);

	/** Atom works best with 256 bytes buffers per channel. */
	uint32 GetDefaultDSPBufferLength() const { return 1024; }

#if ENABLE_ATOM_DEBUG
	/** Get the Atom debugger instance */
	Atom::FAtomDebugger& GetDebugger();
	const Atom::FAtomDebugger& GetDebugger() const;
#endif // ENABLE_ATOM_DEBUG

#if ENABLE_ATOM_MONITOR
	/** Get the Atom Monitor instance */
	FAtomMonitor& GetAtomMonitor();
	const FAtomMonitor& GetAtomMonitor() const;
#endif // ENABLE_ATOM_MONITOR

private:

#if ENABLE_ATOM_DEBUG
	/** Instance of Atom debugger shared across Atim runtimes */
	TUniquePtr<Atom::FAtomDebugger> AtomDebugger;
#endif // ENABLE_ATOM_DEBUG

#if ENABLE_ATOM_MONITOR
	TSharedPtr<FAtomMonitor> AtomMonitor;
#endif // ENABLE_ATOM_MONITOR

	bool InitializeManager();

	/** Creates a handle given the index and a generation value. */
	uint32 GetNewRuntimeID();

	FAtomRuntimeHandle CreateNewRuntime(const FAtomRuntimeParams& InParams);

	// Called exclusively by the FAtomRuntimeHandle copy constructor and assignment operators.
	void IncrementRuntime(FAtomRuntimeId RuntimeID);

	// Called exclusively by the FAtomRuntimeHandle dtor.
	void DecrementRuntime(FAtomRuntimeId RuntimeID, UWorld* InWorld);

	/** Application enters background handler */
	void AppWillEnterBackground();

#if WITH_ENGINE
	/** Delegates used to deal with world switching */
	void OnWorldInitialization(UWorld* InWorld, const UWorld::InitializationValues IVS);
	void OnPostWorldCleanup(UWorld* InWorld, bool bSessionEnded, bool bCleanupResources);
	void OnWorldBeginTearDown(UWorld* InWorld);
	void OnPreWorldFinishDestroy(UWorld* InWorld);
	
	/** Handles used to deal with world switching */
	FDelegateHandle OnPreWorldInitializationHandle;
	FDelegateHandle OnPostWorldCleanupHandle;
	FDelegateHandle OnWorldBeginTearDownHandle;
	FDelegateHandle OnPreWorldFinishDestroyHandle;
#endif
#if WITH_EDITOR
	/** Delegates used to deal with world switching */
	void OnEditorMapChange(uint32);

	/** Handles used to deal with world switching */
	FDelegateHandle OnEditorMapChangeHandle;
#endif

	/** Atom runtime module which creates (old backend) Atom mixer runtime. */
	ICriWarePlatformAtom* AtomRuntimeModule;

	/** The Atom mixer module name. This is the Atom mixer module name to use. E.g. AtomMixerXAudio */
	FString AtomMixerModuleName;

	/** Handle to the main audio device. */
	FAtomRuntimeHandle MainAtomRuntimeHandle;

	struct FAtomRuntimeContainer
	{
		// Singularly owned device.
		// Could be a TUniquePtr if FAtom was not an incomplete type here.
		FAtomRuntime* Runtime;

		// Ref count of FAtomRuntimeHandles referencing this Atom runtime.
		int32 NumberOfHandlesToThisRuntime;

		/** Worlds that have been registered to this Atom runtime. */
		TArray<UWorld*> WorldsUsingThisRuntime;

		/** Whether this Atom runtime can be shared. */
		EAtomRuntimeScope Scope;

		/** Whether this Atom runtime is realtime or not. */
		bool bIsNonRealtime;

		/** Module this was created with. */
		ICriWarePlatformAtom* SpecifiedModule;

		FAtomRuntimeContainer(const FAtomRuntimeParams& InParams, FAtomRuntimeId InRuntimeID, FAtomRuntimeManager* RuntimeManager);
		~FAtomRuntimeContainer();

		FAtomRuntimeContainer(const FAtomRuntimeContainer& Other)
		{
			// We explicitly enforce that we invoke the move instructor.
			// If this was hit, you likely need to call either Devices.Emplace(args) or Devices.Add(DeviceID, MoveTemp(Container));
			checkNoEntry();
		}

		FAtomRuntimeContainer(FAtomRuntimeContainer&& Other);

/*#if INSTRUMENT_AUDIODEVICE_HANDLES
		TMap<uint32, FString> HandleCreationStackWalks;
#endif*/

	private:
		FAtomRuntimeContainer();
	};

	FAtomRuntimeHandle BuildNewHandle(FAtomRuntimeContainer& Container, FAtomRuntimeId RuntimeID, const FAtomRuntimeParams& InParams);

	/**
	 * This function is used to check if we can use an existing Atom runtime.
	 */
	static bool CanUseAtomRuntime(const FAtomRuntimeParams& InParams, const FAtomRuntimeContainer& InContainer);

	/**
	 * Bank of Atom runtimes. Will increase in size as we create new Atom runtimes,
	 */
	TMap<FAtomRuntimeId, FAtomRuntimeContainer> Runtimes;
	mutable FCriticalSection RuntimeMapCriticalSection;

	/** Counter used by GetNewAtomID() to generate a unique ID for a given Atom runtime. */
	uint32 RuntimeIDCounter;

	/** Which audio device is currently active */
	FAtomRuntimeId ActiveAtomRuntimeID;

	/** Dynamic volume map */
	TMap<TTuple<EAtomSoundType, FName>, float> DynamicSoundVolumes;

	/** The Atom resource manager, should always exist. */
	FAtomResourceManager* ResourceManager;

	friend class FAtomRuntimeHandle;

	static FAtomRuntimeManager* Singleton;
};
